<?php
/*
Plugin Name: Newsletter & Cookie Manager
Plugin URI: https://organizator.bg/newsletter-plugin
Description: Управление на настройките за бюлетина и предпочитанията за бисквитки.
Version: 1.2
Author: Anton Stanev
Author URI: https://organizator.bg/
*/

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * ✅ Add a single settings page to the WordPress admin menu
 */
add_action('admin_menu', 'add_newsletter_cookie_settings_page');

function add_newsletter_cookie_settings_page() {
    add_menu_page(
        'Newsletter & Cookies Settings',  // Admin page title
        'Newsletter Cookie',              // Menu title in WordPress sidebar
        'manage_options',                 // Capability (only admins can access)
        'newsletter-cookie-settings',     // Unique slug
        'render_newsletter_cookie_settings', // Callback function
        'dashicons-admin-generic',        // Icon
        100                               // Position in the menu
    );
}

if (!class_exists('NS_Plugin_Updater')) {
    class NS_Plugin_Updater {
        private $api_url = 'https://organizator.bg/api/marketing/plugin/auto-update';
        private $plugin_slug = 'newsletter-cookie';
        private $plugin_version;

        public function __construct() {
            $this->plugin_version = get_file_data(__FILE__, ['Version' => 'Version'])['Version'];

            add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_update']);
            add_filter('plugins_api', [$this, 'plugin_details'], 10, 3);
        }

        public function check_for_update($transient) {
            if (empty($transient->checked)) return $transient;

            $response = wp_remote_get($this->api_url);
            if (is_wp_error($response)) return $transient;

            $data = json_decode(wp_remote_retrieve_body($response));
            if (version_compare($this->plugin_version, $data->version, '<')) {
                $transient->response[$this->plugin_slug . '/' . $this->plugin_slug . '.php'] = (object) [
                    'slug' => $this->plugin_slug,
                    'new_version' => $data->version,
                    'package' => $data->download_url,
                    'tested' => $data->tested,
                    'requires' => $data->requires,
                ];
            }
            return $transient;
        }

        public function plugin_details($result, $action, $args) {
            if ($action !== 'plugin_information' || $args->slug !== $this->plugin_slug) return $result;

            $response = wp_remote_get($this->api_url);
            if (is_wp_error($response)) return $result;

            $data = json_decode(wp_remote_retrieve_body($response));
            return (object) [
                'name' => 'Newsletter & Cookie Consent Plugin',
                'slug' => $this->plugin_slug,
                'version' => $data->version,
                'download_link' => $data->download_url,
                'tested' => $data->tested,
                'requires' => $data->requires,
                'last_updated' => $data->last_updated,
                'sections' => ['description' => 'Персонализиран плъгин за бюлетин и управление на бисквитки.'],
            ];
        }
    }
    new NS_Plugin_Updater();
}

/**
 * ✅ Render the Admin Settings Page
 */
function render_newsletter_cookie_settings() {
    $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'newsletter';
    ?>
    <div class="wrap">
        <h1>Настройки на Newsletter и Бисквитки</h1>

        <h2 class="nav-tab-wrapper">
            <a href="?page=newsletter-cookie-settings&tab=newsletter" class="nav-tab <?php echo $active_tab == 'newsletter' ? 'nav-tab-active' : ''; ?>">Бюлетин</a>
            <a href="?page=newsletter-cookie-settings&tab=cookie" class="nav-tab <?php echo $active_tab == 'cookie' ? 'nav-tab-active' : ''; ?>">Бисквитки</a>
        </h2>

        <form method="post" action="options.php">
            <?php
            if ($active_tab == 'newsletter') {
                include_once __DIR__ . '/newsletter-signup/newsletter-settings.php';
                settings_fields('ns_newsletter_settings');
                do_settings_sections('ns-newsletter-settings');
            } elseif ($active_tab == 'cookie') {
                include_once __DIR__ . '/cookies-preferences/cookie-settings.php';
                settings_fields('ns_cookie_settings');
                do_settings_sections('ns-cookie-settings');
            }
            submit_button('Запази промените');
            ?>
        </form>
    </div>
    <?php
}

/**
 * ✅ Include required PHP files
 * These files handle rendering and logic for the newsletter and cookie modals
 */
if (file_exists(__DIR__ . '/newsletter-signup/newsletter-signup.php')) {
    include_once __DIR__ . '/newsletter-signup/newsletter-signup.php';
}

if (file_exists(__DIR__ . '/cookies-preferences/cookie-preferences-modal.php')) {
    include_once __DIR__ . '/cookies-preferences/cookie-preferences-modal.php';
}

if (file_exists(__DIR__ . '/newsletter-signup/newsletter-settings.php')) {
    include_once __DIR__ . '/newsletter-signup/newsletter-settings.php';
}

if (file_exists(__DIR__ . '/cookies-preferences/cookie-settings.php')) {
    include_once __DIR__ . '/cookies-preferences/cookie-settings.php';
}

/**
 * ✅ Enqueue Admin Panel CSS & JS
 */
function ns_enqueue_admin_assets($hook) {
    if ($hook !== 'toplevel_page_newsletter-cookie-settings') return;

    wp_enqueue_style(
        'style',
        plugins_url('style.css', __FILE__)
    );

    wp_enqueue_script(
        'script',
        plugins_url('script.js', __FILE__),
        [],
        null,
        true
    );
}
add_action('admin_enqueue_scripts', 'ns_enqueue_admin_assets');

/**
 * ✅ Enqueue Frontend CSS & JS
 */
function ns_enqueue_frontend_assets() {
    if (is_admin()) return; // Prevent loading in admin panel

    // Newsletter styles & scripts
    wp_enqueue_style(
        'newsletter-style',
        plugins_url('newsletter-signup/newsletter-style.css', __FILE__)
    );

    wp_enqueue_script(
        'newsletter-script',
        plugins_url('newsletter-signup/newsletter-script.js', __FILE__),
        ['jquery'],
        null,
        true
    );

    // ✅ Cookie Preferences styles & scripts (Handled centrally here)
    wp_enqueue_style(
        'cookie-style',
        plugins_url('cookies-preferences/cookie-preferences.css', __FILE__)
    );

    wp_enqueue_script(
        'cookie-script',
        plugins_url('cookies-preferences/cookie-preferences.js', __FILE__),
        ['jquery'],
        null,
        true
    );

    // Pass necessary data to JavaScript
    wp_localize_script('script', 'nsAjax', [
        'modalDelay' => get_option('ns_modal_delay', 7),
        'privacyUrl' => get_option('ns_privacy_policy_url', '/privacy-policy'),
    ]);
}
add_action('wp_enqueue_scripts', 'ns_enqueue_frontend_assets');
