// 🚀 **Helper Functions (Now shared by both vanilla JS & jQuery)**
function getCookie(name) {
  const cookies = document.cookie.split("; ");
  for (let i = 0; i < cookies.length; i++) {
    let [cookieName, cookieValue] = cookies[i].split("=");
    if (cookieName === name) {
      try {
        return JSON.parse(decodeURIComponent(cookieValue)); // ✅ Decode & parse JSON
      } catch (e) {
        console.error("⚠ Cookie parse error:", e);
        return null;
      }
    }
  }
  return null;
}

function setCookie(name, value, days) {
  let expires = "";
  if (days) {
    const date = new Date();
    date.setTime(date.getTime() + days * 24 * 60 * 60 * 1000);
    expires = "; expires=" + date.toUTCString();
  }
  document.cookie =
    name +
    "=" +
    encodeURIComponent(JSON.stringify(value)) +
    expires +
    "; path=/; SameSite=Lax";
}

// 🚀 **Show Newsletter Modal**
function showNewsletterModal() {
  const modal = document.getElementById("ns-modal-wrapper");
  if (!modal) return console.error("❌ Newsletter modal not found in DOM!");

  modal.classList.remove("hidden");
  modal.style.display = "flex"; // ✅ Ensure it's visible
}

// 🚀 **Ensure modal logic runs after the page loads**
document.addEventListener("DOMContentLoaded", function () {
  console.log("🚀 Checking Newsletter Modal Display Conditions");

  const modal = document.getElementById("ns-modal-wrapper");
  const closeButton = document.getElementById("ns-modal-close");

  if (!modal) {
    return;
  }

  // ✅ Check if modal should appear
  const modalData = getCookie("newsletter_modal");

  if (!modalData || !modalData.closed) {
    setTimeout(() => {
      showNewsletterModal();
    }, 5000); // Default 5s delay
  }

  // 🚀 **Handle Close Button**
  if (closeButton) {
    closeButton.addEventListener("click", function () {
      modal.classList.add("hidden");
      modal.style.display = "none"; // ✅ Ensure it disappears
      setCookie("newsletter_modal", { closed: true }, 60);
    });
  }
});

// 🚀 **jQuery Implementation**
jQuery(document).ready(function ($) {
  if (!isClosed && !isSignedUp) {
    setTimeout(showModal, nsAjax.modalDelay ? nsAjax.modalDelay * 1000 : 7000);
  }

  $("#ns-modal-close, #ns-modal-backdrop").on("click", function () {
    $(".ns-modal-light, #ns-modal-backdrop").fadeOut(function () {
      $(this).addClass("hidden");
      setCookie("ns_modal_closed", "true", 60); // ✅ Store as proper string
    });
  });

  $("#ns-signup-form")
    .off("submit")
    .on("submit", function (e) {
      e.preventDefault();
      const email = $("#email-address").val();
      const companyUuid = $("#ns-company-uuid").val();
      const responseMessage = $("#ns-response-message");

      if (!email || !companyUuid) {
        responseMessage
          .text("Имейл адресът и UUID на компанията са задължителни.")
          .css({ color: "red", display: "block" });
        return;
      }

      const apiUrl = "https://organizator.bg/api/marketing/subscribe";

      $.ajax({
        url: apiUrl,
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify({ email: email, company_uuid: companyUuid }),
        success: function (data) {
          responseMessage
            .text(data.message || "Успешно изпращане!")
            .css({ color: "green", display: "block" });

          setCookie("ns_modal_signed_up", "true", 365); // ✅ Store as proper string

          setTimeout(() => {
            $(".ns-modal-light, #ns-modal-backdrop").fadeOut(() => {
              $(this).addClass("hidden");
            });
          }, 2000);
        },
        error: function (xhr) {
          console.error("Error:", xhr);
          responseMessage
            .text(xhr.responseJSON?.error || "Възникна грешка при изпращане.")
            .css({ color: "red", display: "block" });
        },
      });
    });
});
